# PowerShell script to create desktop shortcut for NeuroConnect v2.0
# Enhanced version with improved functionality and error handling

Write-Host ""
Write-Host "================================================================================================" -ForegroundColor Cyan
Write-Host "             NeuroConnect Therapy Centre Management System v2.0" -ForegroundColor White
Write-Host "                          Desktop Shortcut Creation Script" -ForegroundColor White
Write-Host "================================================================================================" -ForegroundColor Cyan
Write-Host ""

# Check if running as administrator
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")

if (-not $isAdmin) {
    Write-Host "WARNING: Not running as Administrator" -ForegroundColor Yellow
    Write-Host "   Some shortcuts may not be created properly." -ForegroundColor Yellow
    Write-Host ""
}

# Verify required files exist
$requiredFiles = @("NeuroConnect.bat", "login.html", "assets\logo.jpeg")
foreach ($file in $requiredFiles) {
    if (-not (Test-Path $file)) {
        Write-Host "ERROR: Required file not found: $file" -ForegroundColor Red
        Write-Host "   Please ensure you're running this from the NEUROMANAGEMENT folder." -ForegroundColor Red
        Read-Host "Press Enter to exit..."
        exit 1
    }
}

Write-Host "All required files found" -ForegroundColor Green
Write-Host ""

$WshShell = New-Object -comObject WScript.Shell
$DesktopPath = [System.Environment]::GetFolderPath("Desktop")
$ShortcutPath = "$DesktopPath\NeuroConnect Therapy Centre.lnk"

Write-Host "Creating desktop shortcut..." -ForegroundColor Yellow

try {
    $Shortcut = $WshShell.CreateShortcut($ShortcutPath)
    $Shortcut.TargetPath = "$PSScriptRoot\NeuroConnect.bat"
    $Shortcut.WorkingDirectory = $PSScriptRoot
    $Shortcut.Description = "NeuroConnect Therapy Centre Management System v2.0 - Professional offline therapy management"
    $Shortcut.Arguments = ""
    $Shortcut.IconLocation = "$PSScriptRoot\assets\logo.jpeg"
    $Shortcut.WindowStyle = 1  # Normal window
    $Shortcut.Save()
    
    Write-Host "Desktop shortcut created successfully!" -ForegroundColor Green
    Write-Host "   Location: $ShortcutPath" -ForegroundColor Gray
} catch {
    Write-Host "Failed to create desktop shortcut: $($_.Exception.Message)" -ForegroundColor Red
}

Write-Host ""

# Create Start Menu shortcut
Write-Host "Creating Start Menu shortcut..." -ForegroundColor Yellow

try {
    $StartMenuPath = [System.Environment]::GetFolderPath("StartMenu")
    $StartMenuShortcutPath = "$StartMenuPath\Programs\NeuroConnect Therapy Centre.lnk"

    $StartMenuShortcut = $WshShell.CreateShortcut($StartMenuShortcutPath)
    $StartMenuShortcut.TargetPath = "$PSScriptRoot\NeuroConnect.bat"
    $StartMenuShortcut.WorkingDirectory = $PSScriptRoot
    $StartMenuShortcut.Description = "NeuroConnect Therapy Centre Management System v2.0"
    $StartMenuShortcut.IconLocation = "$PSScriptRoot\assets\logo.jpeg"
    $StartMenuShortcut.WindowStyle = 1
    $StartMenuShortcut.Save()
    
    Write-Host "Start Menu shortcut created successfully!" -ForegroundColor Green
    Write-Host "   Location: $StartMenuShortcutPath" -ForegroundColor Gray
} catch {
    Write-Host "Start Menu shortcut creation failed: $($_.Exception.Message)" -ForegroundColor Yellow
    Write-Host "   This is normal if not running as Administrator." -ForegroundColor Gray
}

Write-Host ""
Write-Host "================================================================================================" -ForegroundColor Cyan
Write-Host "                                Setup Complete!" -ForegroundColor White
Write-Host "================================================================================================" -ForegroundColor Cyan
Write-Host ""
Write-Host "NeuroConnect shortcuts have been created successfully!" -ForegroundColor Green
Write-Host ""
Write-Host "How to launch NeuroConnect:" -ForegroundColor White
Write-Host "  1. Double-click 'NeuroConnect Therapy Centre' icon on your desktop" -ForegroundColor Gray
Write-Host "  2. Search 'NeuroConnect' in Start Menu" -ForegroundColor Gray
Write-Host "  3. Run NeuroConnect.bat manually from this folder" -ForegroundColor Gray
Write-Host ""
Write-Host "System Features Available:" -ForegroundColor White
Write-Host "  - Enhanced Therapy Scheduling with detailed views" -ForegroundColor Green
Write-Host "  - Role-based Authentication (Admin/User)" -ForegroundColor Green
Write-Host "  - Real NeuroConnect Directors Database" -ForegroundColor Green
Write-Host "  - Patient and Staff Management" -ForegroundColor Green
Write-Host "  - Financial Tracking in Ksh Currency" -ForegroundColor Green
Write-Host "  - Stock Control and Inventory" -ForegroundColor Green
Write-Host "  - Comprehensive Reporting" -ForegroundColor Green
Write-Host "  - Fully Offline Operation" -ForegroundColor Green
Write-Host ""
Write-Host "Login Credentials:" -ForegroundColor White
Write-Host "  Admin: admin / admin123 (Full access)" -ForegroundColor Cyan
Write-Host "  User:  user / user123   (Limited access)" -ForegroundColor Cyan
Write-Host ""
Write-Host "For support and documentation, see README.md" -ForegroundColor Gray
Write-Host ""

Read-Host "Press Enter to continue"
